﻿# NebulaCanvas Deployment Guide

NebulaCanvas is a lightweight PHP AI image generator designed for shared hosting (Namecheap, cPanel, etc.). It ships with a mock provider so you can launch a fully working demo without inference costs and swap in a remote model when ready.

## Stack at a glance

- PHP 8.1+, no framework, no build step on the server
- SQLite for jobs, rate limiting, and metadata (single file)
- Tailwind-style CSS precompiled into `public_html/assets/styles.css`
- Minimal vanilla JS for UX enhancements only
- Files live under `public_html/` for direct upload via cPanel File Manager

## Local development

1. Install PHP 8.1+ and (optionally) Composer / PHPUnit.
2. Copy `.env` values by editing `config.php` directly (no runtime env needed).
3. Start the PHP dev server:
   ```bash
   php -S localhost:8000 -t public_html
   ```
4. Visit [http://localhost:8000](http://localhost:8000) to explore the site.

### Optional: recompile CSS

- Install Tailwind CLI: `npm install -g tailwindcss`
- Run `make css` (uses `styles/tailwind.css` as the input source). A precompiled `public_html/assets/styles.css` is already provided, so this step is only for designers iterating on the theme.

## Configuration (`config.php`)

| Key | Description | Default |
| --- | --- | --- |
| `SITE_NAME` | Display name in headers and metadata | `NebulaCanvas` |
| `SITE_URL` | Public base URL used for canonical tags, sitemap, robots | `https://example.com` |
| `CRON_TOKEN` | Shared secret to protect `/api/cleanup.php` | `set-a-secure-token` |
| `RATE_LIMIT_MAX` | Requests per sliding window per IP | `10` |
| `RATE_LIMIT_WINDOW_SEC` | Sliding window size (seconds) | `60` |
| `DB_PATH` | SQLite database path | `public_html/storage/meta/app.sqlite` |
| `STORAGE_DIR` | Directory for generated binaries | `public_html/storage/images` |
| `WATERMARK_ENABLED` / `WATERMARK_TEXT` | Toggle & text for subtle watermarking | Enabled / `Generated @ NebulaCanvas` |
| `PROVIDER` | `mock` or `remote_http` | `mock` |
| `REMOTE_HTTP_ENDPOINT` | HTTPS endpoint for real diffusion models | - |
| `REMOTE_HTTP_API_KEY` | Bearer token for remote provider | - |
| `REMOTE_HTTP_TIMEOUT` | Seconds before remote request times out | `60` |

Update `SITE_URL` and `CRON_TOKEN` immediately after deploying.

## Provider modes

- **Mock (default):** Generates tasteful placeholder art with PHP/GD instantly and costs nothing.
- **Remote HTTP:** Switch by setting `PROVIDER=remote_http`, `REMOTE_HTTP_ENDPOINT`, and optionally `REMOTE_HTTP_API_KEY` in `config.php`. The remote endpoint must return `{ "image": base64, "mime": "image/png" }`.

## Image lifecycle & auto-deletion

1. `/api/generate.php` validates input, enforces the SQLite-backed rate limiter, and stores metadata via `ImageJobService`.
2. Files land under `public_html/storage/images` with permissions `0664`. Direct access is blocked by `.htaccess`.
3. `deleteAt` (createdAt + exactly 300000 ms) is persisted and enforced everywhere:
   - `/api/image.php` checks TTL on every request and returns HTTP 410 with `{ "reason": "expired" }` after expiry.
   - `/api/cleanup.php?token=CRON_TOKEN` sweeps stale rows and files; configure a cron job every minute.

### Required cron (cPanel example)

```
* * * * * /usr/bin/curl -fsS https://YOUR_DOMAIN/api/cleanup.php?token=YOUR_TOKEN >/dev/null
```

## Security & privacy

- `.htaccess` denies direct access to `storage/` and forces security headers.
- `/api/image.php` ships with strict `Cache-Control: no-store` and `X-Content-Type-Options: nosniff`.
- Prompts are capped at 500 characters and simple keyword filtering blocks obvious NSFW requests.
- No analytics by default. If you add a privacy-friendly script, place it in `partials/meta.php` and update the privacy policy.

## Running tests

### Unit tests (PHPUnit 10+ recommended)

```bash
composer global require phpunit/phpunit # if not already installed
phpunit --configuration phpunit.xml
```

Covers TTL math, input validation, rate limiting, and cleanup.

### E2E smoke (uses PHP cURL, expects the app running locally)

```bash
BASE_URL=http://localhost:8000 php tests/e2e.php
```

> NOTE: The script waits the full 5-minute TTL (default wait 305 seconds). Override with `WAIT_SECONDS=10` if you temporarily shorten the TTL during development.

Generated images are saved under `tests/tmp/` for inspection and removed on next test run.

## Deployment (Namecheap / cPanel workflow)

1. Run unit tests locally.
2. Zip the repository root (keeping the `public_html/` folder structure intact).
3. Upload via cPanel File Manager and extract into your hosting account.
4. Ensure `public_html/storage/images` and `public_html/storage/meta` are writable (755 or 775 depending on host).
5. Edit `config.php` in-place with your production values.
6. Configure the cron job for `/api/cleanup.php` (see above).
7. Clear any cached assets (if your host uses caching) to ensure the precompiled CSS loads.

## Operator notes

- **Cron safety:** If the cron fails, images still expire via on-demand checks; the cron is a safety net.
- **Provider toggles:** `PROVIDER=mock` keeps the service free. Switch to `remote_http` with your inference endpoint when ready. The UI remains unchanged.
- **Storage permissions:** Ensure `storage/images` retains execute permissions so PHP can write files, but `.htaccess` keeps them private.
- **Bandwidth:** Images stream via PHP with `no-store` headers. Encourage downloads rather than hotlinking to manage bandwidth.
- **Watermarking:** Disable by setting `WATERMARK_ENABLED=false` or editing `WATERMARK_TEXT`.
- **Error visibility:** Logs land in `public_html/storage/meta/app.log`. Rotate or download periodically if hosting quotas are tight.

## Lighthouse budget

A sample Lighthouse performance budget is available at `docs/lighthouse-budget.json`. Run:

```bash
lighthouse http://localhost:8000 --budgets-path=docs/lighthouse-budget.json
```

This keeps First Contentful Paint under 1.8s and Total Blocking Time under 150ms on mid-tier devices.