﻿<?php

namespace App\Lib;

use PDO;

class RateLimiter
{
    public static function attempt(string $ip): array
    {
        $config = Env::all();
        $max = (int)($config['RATE_LIMIT_MAX'] ?? 10);
        $windowSec = (int)($config['RATE_LIMIT_WINDOW_SEC'] ?? 60);
        $windowMs = $windowSec * 1000;
        $now = Time::nowMs();

        $pdo = Db::conn();
        $pdo->beginTransaction();
        $stmt = $pdo->prepare('SELECT windowStart, count FROM rate_limits WHERE ip = :ip');
        $stmt->execute([':ip' => $ip]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($row) {
            $windowStart = (int)$row['windowStart'];
            $count = (int)$row['count'];
            if ($now - $windowStart >= $windowMs) {
                $windowStart = $now;
                $count = 0;
            }
            if ($count >= $max) {
                $retryAfter = (int) ceil(($windowStart + $windowMs - $now) / 1000);
                $pdo->commit();
                return [
                    'allowed' => false,
                    'retryAfter' => max($retryAfter, 1),
                ];
            }
            $count++;
            $update = $pdo->prepare('UPDATE rate_limits SET count = :count, windowStart = :windowStart, updatedAt = :updatedAt WHERE ip = :ip');
            $update->execute([
                ':count' => $count,
                ':windowStart' => $windowStart,
                ':updatedAt' => $now,
                ':ip' => $ip,
            ]);
        } else {
            $windowStart = $now;
            $count = 1;
            $insert = $pdo->prepare('INSERT INTO rate_limits (ip, windowStart, count, updatedAt) VALUES (:ip, :windowStart, :count, :updatedAt)');
            $insert->execute([
                ':ip' => $ip,
                ':windowStart' => $windowStart,
                ':count' => $count,
                ':updatedAt' => $now,
            ]);
        }
        $pdo->commit();

        return [
            'allowed' => true,
            'retryAfter' => 0,
        ];
    }
}