﻿<?php

namespace App\Lib;

use RuntimeException;

class StorageService
{
    public static function store(string $imageId, string $binary, string $mime): string
    {
        $config = Env::all();
        $baseDir = rtrim($config['STORAGE_DIR'], '/');
        if (!is_dir($baseDir)) {
            mkdir($baseDir, 0775, true);
        }

        $extension = self::extensionFromMime($mime);
        $finalPath = $baseDir . '/' . $imageId . '.' . $extension;
        $tmpPath = $finalPath . '.tmp';

        $payload = $binary;
        if (!empty($config['WATERMARK_ENABLED'])) {
            $payload = self::watermark($binary, $mime, (string)($config['WATERMARK_TEXT'] ?? ''));
        }

        if (file_put_contents($tmpPath, $payload) === false) {
            throw new RuntimeException('Unable to write image to disk.');
        }

        if (!rename($tmpPath, $finalPath)) {
            @unlink($tmpPath);
            throw new RuntimeException('Unable to finalize image write.');
        }

        @chmod($finalPath, 0664);

        return $finalPath;
    }

    public static function delete(string $filePath): void
    {
        if ($filePath && file_exists($filePath)) {
            @unlink($filePath);
        }
    }

    private static function watermark(string $binary, string $mime, string $text): string
    {
        if ($text === '') {
            return $binary;
        }
        if (!extension_loaded('gd')) {
            return $binary;
        }
        $image = imagecreatefromstring($binary);
        if (!$image) {
            return $binary;
        }
        $width = imagesx($image);
        $height = imagesy($image);
        $color = imagecolorallocatealpha($image, 255, 255, 255, 70);
        $fontSize = 3; // built-in font index
        $textWidth = imagefontwidth($fontSize) * strlen($text);
        $textHeight = imagefontheight($fontSize);
        $x = max(2, $width - $textWidth - 6);
        $y = max(2, $height - $textHeight - 6);
        imagestring($image, $fontSize, $x, $y, $text, $color);

        ob_start();
        switch ($mime) {
            case 'image/jpeg':
                imagejpeg($image, null, 92);
                break;
            case 'image/webp':
                imagewebp($image, null, 90);
                break;
            default:
                imagepng($image);
        }
        $output = ob_get_clean();
        imagedestroy($image);
        return $output ?: $binary;
    }

    private static function extensionFromMime(string $mime): string
    {
        return match ($mime) {
            'image/jpeg' => 'jpg',
            'image/webp' => 'webp',
            'image/png' => 'png',
            default => 'png',
        };
    }
}