﻿<?php

namespace App\Lib\Providers;

use App\Lib\Env;
use App\Lib\ImageProvider;
use App\Lib\ImageProviderResult;
use App\Lib\Time;

class MockProvider implements ImageProvider
{
    public function generate(array $payload): ImageProviderResult
    {
        $width = max(256, min((int)($payload['width'] ?? 768), 1024));
        $height = max(256, min((int)($payload['height'] ?? 768), 1024));
        $prompt = trim((string)($payload['prompt'] ?? 'Calming abstract gradient'));
        $mime = 'image/png';

        $binary = Env::get('MOCK_PLACEHOLDER_PATH') ? $this->loadFromFile() : $this->generatePlaceholder($width, $height, $prompt);

        return new ImageProviderResult($mime, $binary, [
            'provider' => 'mock',
            'prompt' => $prompt,
            'width' => $width,
            'height' => $height,
            'generatedAt' => Time::nowMs(),
        ]);
    }

    private function loadFromFile(): string
    {
        $path = Env::get('MOCK_PLACEHOLDER_PATH');
        if ($path && file_exists($path)) {
            return file_get_contents($path) ?: '';
        }
        return '';
    }

    private function generatePlaceholder(int $width, int $height, string $prompt): string
    {
        if (!extension_loaded('gd')) {
            return base64_decode('iVBORw0KGgoAAAANSUhEUgAAAEAAAABACAQAAAAAYLlVAAAAnklEQVR42u3WwQ3CMBQF0MkKDCEDxGcZgLkmEyrAgNsq4CPGK/cHBt9nuSUXGKmzMEyXoZPku0tJXoST6JB2nNuhaIlr4Px59T/VPzD7DCwyIB9GMv9yZOP3rf+YMGsKRNR9qBRL6ka0DAtJFrSAMSiKehQWiQa0gDEoinsUFlEGrSA8Sga70gLEoCu9IAxKIr7FBZRBq0gPEoGu9ICxKA3cZp2g0f1yv+uKOnHBb2D9bwZTtqL7WGNhRF0NcFEP9A/DU0YaiuEZAAAAAElFTkSuQmCC');
        }

        $im = imagecreatetruecolor($width, $height);
        imagealphablending($im, false);
        $bg = imagecolorallocate($im, 11, 18, 32);
        imagefill($im, 0, 0, $bg);
        imagealphablending($im, true);

        $start = [93, 168, 255];
        $end = [74, 222, 128];
        for ($y = 0; $y < $height; $y++) {
            $ratio = $y / max($height - 1, 1);
            $r = (int)($start[0] * (1 - $ratio) + $end[0] * $ratio);
            $g = (int)($start[1] * (1 - $ratio) + $end[1] * $ratio);
            $b = (int)($start[2] * (1 - $ratio) + $end[2] * $ratio);
            $lineColor = imagecolorallocatealpha($im, $r, $g, $b, 80);
            imageline($im, 0, $y, $width, $y, $lineColor);
        }

        $accent = imagecolorallocatealpha($im, 74, 222, 128, 90);
        imagefilledellipse($im, (int)($width * 0.25), (int)($height * 0.4), (int)($width * 0.6), (int)($height * 0.6), $accent);
        imagefilledellipse($im, (int)($width * 0.7), (int)($height * 0.55), (int)($width * 0.5), (int)($height * 0.5), imagecolorallocatealpha($im, 93, 168, 255, 100));

        $colorText = imagecolorallocate($im, 230, 238, 248);
        $font = 4;
        $wrapped = wordwrap($prompt, 28, "\n");
        $lines = explode("\n", $wrapped);
        $textHeight = count($lines) * (imagefontheight($font) + 4);
        $y = (int)(($height - $textHeight) / 2);
        foreach ($lines as $line) {
            $textWidth = strlen($line) * imagefontwidth($font);
            $x = max((int)(($width - $textWidth) / 2), 10);
            imagestring($im, $font, $x, $y, $line, $colorText);
            $y += imagefontheight($font) + 4;
        }

        $stamp = imagecolorallocatealpha($im, 147, 164, 189, 70);
        imagestring($im, 2, 10, $height - 20, 'Mock Provider', $stamp);

        ob_start();
        imagesavealpha($im, true);
        imagepng($im);
        $data = ob_get_clean();
        imagedestroy($im);
        return $data ?: '';
    }
}