﻿<?php

namespace App\Lib\Providers;

use App\Lib\Env;
use App\Lib\ImageProvider;
use App\Lib\ImageProviderResult;
use App\Lib\Time;
use RuntimeException;

class RemoteHTTPProvider implements ImageProvider
{
    public function generate(array $payload): ImageProviderResult
    {
        $endpoint = Env::get('REMOTE_HTTP_ENDPOINT');
        if (!$endpoint) {
            throw new RuntimeException('REMOTE_HTTP_ENDPOINT not configured.');
        }

        $options = [
            'prompt' => trim((string)($payload['prompt'] ?? '')),
            'width' => max(256, min((int)($payload['width'] ?? 768), 1024)),
            'height' => max(256, min((int)($payload['height'] ?? 768), 1024)),
            'steps' => max(10, min((int)($payload['steps'] ?? 30), 150)),
            'guidance' => max(1, min((float)($payload['guidance'] ?? 7.5), 30)),
            'seed' => $payload['seed'] ?? null,
        ];

        $body = json_encode($options, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
        $timeout = (int)Env::get('REMOTE_HTTP_TIMEOUT', 60);

        $headers = [
            'Content-Type: application/json',
        ];
        $apiKey = Env::get('REMOTE_HTTP_API_KEY');
        if ($apiKey) {
            $headers[] = 'Authorization: Bearer ' . $apiKey;
        }

        $ch = curl_init($endpoint);
        curl_setopt_array($ch, [
            CURLOPT_POST => true,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_POSTFIELDS => $body,
            CURLOPT_TIMEOUT => $timeout,
        ]);
        $response = curl_exec($ch);
        if ($response === false) {
            $error = curl_error($ch);
            curl_close($ch);
            throw new RuntimeException('Remote provider error: ' . $error);
        }
        $status = curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
        curl_close($ch);

        if ($status >= 400) {
            throw new RuntimeException('Remote provider returned HTTP ' . $status . ' body: ' . $response);
        }

        $decoded = json_decode($response, true);
        if (!is_array($decoded)) {
            throw new RuntimeException('Remote provider response not JSON.');
        }

        $imageBase64 = $decoded['image'] ?? null;
        $mime = $decoded['mime'] ?? 'image/png';
        if (!$imageBase64) {
            throw new RuntimeException('Remote provider response missing image key.');
        }

        $binary = base64_decode($imageBase64, true);
        if ($binary === false) {
            throw new RuntimeException('Remote provider image payload invalid.');
        }

        return new ImageProviderResult($mime, $binary, [
            'provider' => 'remote_http',
            'endpoint' => $endpoint,
            'request' => $options,
            'raw' => $decoded,
            'generatedAt' => Time::nowMs(),
        ]);
    }
}