﻿(function () {
    const qs = (sel, ctx = document) => ctx.querySelector(sel);
    const qsa = (sel, ctx = document) => Array.from(ctx.querySelectorAll(sel));
    const form = qs('#generate-form');
    if (!form) return;

    const promptField = qs('#prompt');
    const statusEl = qs('#result-status');
    const resultSection = qs('#result');
    const resultImg = qs('#result-image');
    const skeleton = qs('.skeleton');
    const resultPrompt = qs('#result-prompt');
    const resultDimensions = qs('#result-dimensions');
    const countdownText = qs('#countdown-text');
    const ring = qs('#ring-progress');
    const downloadLink = qs('#download-link');
    const copyLinkBtn = qs('#copy-link');
    const regenBtn = qs('#regenerate');
    const expiryNotice = qs('#expiry-notice');
    const formFoot = qs('.form-foot');
    const navToggle = qs('.nav-toggle');
    const navMenu = qs('#nav-menu');

    if (navToggle && navMenu) {
        navToggle.addEventListener('click', () => {
            const expanded = navToggle.getAttribute('aria-expanded') === 'true';
            navToggle.setAttribute('aria-expanded', String(!expanded));
            navMenu.dataset.open = (!expanded).toString();
        });
    }

    qsa('.prompt-chip').forEach((chip) => {
        chip.addEventListener('click', () => {
            promptField.value = chip.dataset.prompt || '';
            promptField.focus();
        });
    });

    let countdownTimer;
    function startCountdown(deleteAt) {
        const totalMs = deleteAt - Date.now();
        if (totalMs <= 0) {
            markExpired();
            return;
        }
        updateCountdown(deleteAt);
        if (countdownTimer) window.clearInterval(countdownTimer);
        countdownTimer = window.setInterval(() => updateCountdown(deleteAt), 1000);
    }

    function updateCountdown(deleteAt) {
        const remaining = Math.max(0, deleteAt - Date.now());
        const total = 300000;
        const progress = Math.max(0, Math.min(1, remaining / total));
        const minutes = Math.floor(remaining / 60000);
        const seconds = Math.floor((remaining % 60000) / 1000);
        countdownText.textContent = `${String(minutes).padStart(2, '0')}:${String(seconds).padStart(2, '0')}`;
        ring.style.strokeDashoffset = String(100 - progress * 100);
        if (remaining === 0) {
            markExpired();
        }
    }

    function markExpired() {
        statusEl.textContent = 'Expired';
        expiryNotice.hidden = false;
        if (countdownTimer) window.clearInterval(countdownTimer);
    }

    async function pollProgress(jobId) {
        try {
            const res = await fetch(`api/progress.php?jobId=${encodeURIComponent(jobId)}`);
            if (!res.ok) throw new Error('progress failed');
            const data = await res.json();
            statusEl.textContent = data.status;
            if (data.status === 'failed') {
                formFoot.textContent = data.error || 'The image failed to generate.';
                formFoot.classList.add('error');
                return;
            }
            if (data.status !== 'done') {
                window.setTimeout(() => pollProgress(jobId), 1000);
            }
            if (typeof data.remainingMs === 'number') {
                updateCountdown(Date.now() + data.remainingMs);
            }
        } catch (err) {
            console.error(err);
        }
    }

    form.addEventListener('submit', async (event) => {
        event.preventDefault();
        formFoot.textContent = '';
        formFoot.classList.remove('error');
        expiryNotice.hidden = true;
        statusEl.textContent = 'Working...';
        resultSection.hidden = false;
        skeleton.hidden = false;
        resultImg.hidden = true;

        const formData = new FormData(form);
        const payload = Object.fromEntries(formData.entries());
        Object.keys(payload).forEach((key) => {
            if (payload[key] === '') delete payload[key];
        });

        try {
            const res = await fetch('api/generate.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(payload),
            });
            const data = await res.json();
            if (!res.ok) {
                throw new Error(data.error || 'Failed to generate image.');
            }
            const imageUrl = `api/image.php?id=${encodeURIComponent(data.imageId)}`;
            resultImg.onload = () => {
                skeleton.hidden = true;
                resultImg.hidden = false;
                statusEl.textContent = 'Ready';
                startCountdown(data.deleteAt);
            };
            resultImg.src = imageUrl;
            resultImg.alt = `Generated artwork for prompt: ${promptField.value}`;
            resultPrompt.textContent = promptField.value;
            resultDimensions.textContent = `${payload.width || 768} x ${payload.height || 768}`;
            downloadLink.href = imageUrl;
            downloadLink.download = `${data.imageId}.png`;
            const absoluteLink = new URL(imageUrl, window.location.href).href;
            copyLinkBtn.dataset.link = absoluteLink;
            statusEl.textContent = 'Rendering...';
            pollProgress(data.jobId);
        } catch (error) {
            console.error(error);
            formFoot.textContent = error.message;
            formFoot.classList.add('error');
            statusEl.textContent = 'Error';
        }
    });

    copyLinkBtn?.addEventListener('click', async () => {
        const link = copyLinkBtn.dataset.link;
        if (!link) return;
        try {
            await navigator.clipboard.writeText(link);
            copyLinkBtn.textContent = 'Copied!';
            window.setTimeout(() => (copyLinkBtn.textContent = 'Copy link'), 3000);
        } catch (err) {
            console.error(err);
        }
    });

    regenBtn?.addEventListener('click', () => {
        window.scrollTo({ top: 0, behavior: 'smooth' });
        promptField.focus();
    });
})();