﻿<?php

namespace App\Lib;

class ContentRepository
{
    private string $basePath;

    public function __construct()
    {
        $this->basePath = dirname(__DIR__) . '/content';
    }

    public function blogIndex(): array
    {
        $dir = $this->basePath . '/blog';
        if (!is_dir($dir)) {
            return [];
        }
        $posts = [];
        foreach (glob($dir . '/*.md') as $file) {
            $parsed = $this->parseBlogFile($file);
            if ($parsed) {
                $posts[] = $parsed;
            }
        }
        usort($posts, fn($a, $b) => strcmp($b['published_at'], $a['published_at']));
        return $posts;
    }

    public function findBlog(string $slug): ?array
    {
        $dir = $this->basePath . '/blog';
        $paths = [
            $dir . '/' . $slug . '.md',
            $dir . '/' . $slug . '.markdown',
        ];
        foreach ($paths as $path) {
            if (file_exists($path)) {
                return $this->parseBlogFile($path);
            }
        }
        return null;
    }

    private function parseBlogFile(string $path): ?array
    {
        $contents = file_get_contents($path);
        if ($contents === false) {
            return null;
        }
        $parts = preg_split('/^-{3,}\s*$/m', $contents);
        if (!$parts || count($parts) < 3) {
            return null;
        }
        $metaRaw = trim($parts[1]);
        $bodyRaw = trim($parts[2]);
        $meta = $this->parseFrontMatter($metaRaw);
        $slug = $meta['slug'] ?? pathinfo($path, PATHINFO_FILENAME);
        $title = $meta['title'] ?? ucfirst(str_replace('-', ' ', $slug));
        $publishedAt = $meta['published_at'] ?? date('c');
        $updatedAt = $meta['updated_at'] ?? $publishedAt;
        $excerpt = $meta['excerpt'] ?? substr(strip_tags($bodyRaw), 0, 140);
        $bodyHtml = $this->markdownToHtml($bodyRaw);

        return [
            'slug' => $slug,
            'title' => $title,
            'published_at' => $publishedAt,
            'updated_at' => $updatedAt,
            'excerpt' => $excerpt,
            'html' => $bodyHtml,
        ];
    }

    private function parseFrontMatter(string $input): array
    {
        $lines = preg_split('/\r?\n/', $input);
        $meta = [];
        foreach ($lines as $line) {
            if (strpos($line, ':') === false) {
                continue;
            }
            [$key, $value] = array_map('trim', explode(':', $line, 2));
            $meta[$key] = $value;
        }
        return $meta;
    }

    private function markdownToHtml(string $markdown): string
    {
        $html = '';
        $escape = static fn(string $value): string => htmlspecialchars($value, ENT_QUOTES, 'UTF-8');
        $blocks = preg_split('/\r?\n\r?\n/', trim($markdown));
        foreach ($blocks as $block) {
            $trimmed = trim($block);
            if ($trimmed === '') {
                continue;
            }
            if (str_starts_with($trimmed, '## ')) {
                $content = trim(substr($trimmed, 3));
                $html .= '<h2>' . $escape($content) . '</h2>';
            } elseif (str_starts_with($trimmed, '# ')) {
                $content = trim(substr($trimmed, 2));
                $html .= '<h1>' . $escape($content) . '</h1>';
            } elseif (preg_match('/^-\s+/m', $trimmed)) {
                $html .= '<ul>';
                $items = preg_split('/\r?\n/', $trimmed);
                foreach ($items as $item) {
                    $html .= '<li>' . $escape(ltrim($item, '- ')) . '</li>';
                }
                $html .= '</ul>';
            } else {
                $html .= '<p>' . $escape($trimmed) . '</p>';
            }
        }
        return $html;
    }
}