﻿<?php

namespace App\Lib;

use PDO;
use PDOException;

class Db
{
    private static ?PDO $pdo = null;

    public static function conn(): PDO
    {
        if (self::$pdo instanceof PDO) {
            return self::$pdo;
        }

        $config = Env::all();
        $path = $config['DB_PATH'];
        $dir = dirname($path);
        if (!is_dir($dir)) {
            mkdir($dir, 0775, true);
        }

        $dsn = 'sqlite:' . $path;
        $options = [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_TIMEOUT => 5,
        ];

        $pdo = new PDO($dsn, null, null, $options);
        $pdo->exec('PRAGMA foreign_keys = ON');
        $pdo->exec('PRAGMA journal_mode = WAL');
        $pdo->exec('PRAGMA busy_timeout = 5000');
        self::$pdo = $pdo;
        self::migrate();

        return self::$pdo;
    }

    private static function migrate(): void
    {
        $pdo = self::$pdo;
        if (!$pdo) {
            throw new PDOException('Database connection not initialised.');
        }

        $pdo->exec('CREATE TABLE IF NOT EXISTS images (
            id TEXT PRIMARY KEY,
            createdAt INTEGER NOT NULL,
            deleteAt INTEGER NOT NULL,
            prompt TEXT NOT NULL,
            width INTEGER NOT NULL,
            height INTEGER NOT NULL,
            filePath TEXT NOT NULL,
            mime TEXT NOT NULL,
            status TEXT CHECK(status IN (\'queued\', \'working\', \'done\', \'failed\')) NOT NULL DEFAULT \'done\'
        )');

        $pdo->exec('CREATE TABLE IF NOT EXISTS jobs (
            id TEXT PRIMARY KEY,
            imageId TEXT NOT NULL,
            status TEXT CHECK(status IN (\'queued\', \'working\', \'done\', \'failed\')) NOT NULL,
            progress INTEGER DEFAULT 0,
            error TEXT,
            createdAt INTEGER NOT NULL,
            updatedAt INTEGER NOT NULL,
            FOREIGN KEY(imageId) REFERENCES images(id) ON DELETE CASCADE
        )');

        $pdo->exec('CREATE INDEX IF NOT EXISTS idx_jobs_status ON jobs(status)');

        $pdo->exec('CREATE TABLE IF NOT EXISTS rate_limits (
            ip TEXT NOT NULL,
            windowStart INTEGER NOT NULL,
            count INTEGER NOT NULL,
            updatedAt INTEGER NOT NULL,
            PRIMARY KEY (ip)
        )');

        $pdo->exec('CREATE TABLE IF NOT EXISTS settings (
            key TEXT PRIMARY KEY,
            value TEXT
        )');
    }
}