﻿<?php

namespace App\Lib;

use RuntimeException;

class ImageJobService
{
    public const TTL_MS = 300000; // 5 minutes

    public function persist(array $input, ImageProviderResult $result, ?int $nowMs = null): array
    {
        if (!$result->binary) {
            throw new RuntimeException('Image payload empty.');
        }
        $now = $nowMs ?? Time::nowMs();
        $deleteAt = self::computeDeleteAt($now);
        $imageId = Id::ulid();
        $jobId = Id::ulid();
        $path = StorageService::store($imageId, $result->binary, $result->mime);

        $pdo = Db::conn();
        $pdo->beginTransaction();
        $stmt = $pdo->prepare('INSERT INTO images (id, createdAt, deleteAt, prompt, width, height, filePath, mime, status) VALUES (:id, :createdAt, :deleteAt, :prompt, :width, :height, :filePath, :mime, :status)');
        $stmt->execute([
            ':id' => $imageId,
            ':createdAt' => $now,
            ':deleteAt' => $deleteAt,
            ':prompt' => $input['prompt'],
            ':width' => $input['width'],
            ':height' => $input['height'],
            ':filePath' => $path,
            ':mime' => $result->mime,
            ':status' => 'done',
        ]);

        $jobStmt = $pdo->prepare('INSERT INTO jobs (id, imageId, status, progress, createdAt, updatedAt) VALUES (:id, :imageId, :status, :progress, :createdAt, :updatedAt)');
        $jobStmt->execute([
            ':id' => $jobId,
            ':imageId' => $imageId,
            ':status' => 'done',
            ':progress' => 100,
            ':createdAt' => $now,
            ':updatedAt' => $now,
        ]);
        $pdo->commit();

        return [
            'imageId' => $imageId,
            'jobId' => $jobId,
            'deleteAt' => $deleteAt,
            'filePath' => $path,
            'createdAt' => $now,
        ];
    }

    public static function computeDeleteAt(int $createdAt): int
    {
        return $createdAt + self::TTL_MS;
    }
}