﻿<?php

namespace App\Lib;

use InvalidArgumentException;

class SchemaField
{
    public string $type;
    public bool $required = false;
    public mixed $default = null;
    public ?int $min = null;
    public ?int $max = null;
    public ?float $minFloat = null;
    public ?float $maxFloat = null;
    public bool $nullable = false;

    public function __construct(string $type)
    {
        $this->type = $type;
    }

    public function required(): self
    {
        $this->required = true;
        return $this;
    }

    public function nullable(): self
    {
        $this->nullable = true;
        return $this;
    }

    public function default(mixed $value): self
    {
        $this->default = $value;
        return $this;
    }

    public function min(int $value): self
    {
        $this->min = $value;
        return $this;
    }

    public function max(int $value): self
    {
        $this->max = $value;
        return $this;
    }

    public function minFloat(float $value): self
    {
        $this->minFloat = $value;
        return $this;
    }

    public function maxFloat(float $value): self
    {
        $this->maxFloat = $value;
        return $this;
    }
}

class Schema
{
    private array $fields;

    public function __construct(array $fields)
    {
        $this->fields = $fields;
    }

    public function parse(array $input): array
    {
        $output = [];
        foreach ($this->fields as $key => $field) {
            $value = $input[$key] ?? null;
            if ($value === null) {
                if ($field->required && !$field->nullable) {
                    throw new InvalidArgumentException($key . ' is required.');
                }
                $value = $field->default;
                if ($value === null && !$field->nullable) {
                    continue;
                }
            }

            switch ($field->type) {
                case 'string':
                    if ($value === null && $field->nullable) {
                        $output[$key] = null;
                        break;
                    }
                    $value = trim((string)$value);
                    $len = strlen($value);
                    if ($field->min !== null && $len < $field->min) {
                        throw new InvalidArgumentException($key . ' must be at least ' . $field->min . ' characters.');
                    }
                    if ($field->max !== null && $len > $field->max) {
                        throw new InvalidArgumentException($key . ' must be at most ' . $field->max . ' characters.');
                    }
                    $output[$key] = $value;
                    break;
                case 'int':
                    if ($value === null && $field->nullable) {
                        $output[$key] = null;
                        break;
                    }
                    if (!is_numeric($value)) {
                        throw new InvalidArgumentException($key . ' must be an integer.');
                    }
                    $value = (int)$value;
                    if ($field->min !== null && $value < $field->min) {
                        $value = $field->min;
                    }
                    if ($field->max !== null && $value > $field->max) {
                        $value = $field->max;
                    }
                    $output[$key] = $value;
                    break;
                case 'float':
                    if ($value === null && $field->nullable) {
                        $output[$key] = null;
                        break;
                    }
                    if (!is_numeric($value)) {
                        throw new InvalidArgumentException($key . ' must be numeric.');
                    }
                    $value = (float)$value;
                    if ($field->minFloat !== null && $value < $field->minFloat) {
                        $value = $field->minFloat;
                    }
                    if ($field->maxFloat !== null && $value > $field->maxFloat) {
                        $value = $field->maxFloat;
                    }
                    $output[$key] = $value;
                    break;
                case 'mixed':
                    $output[$key] = $value;
                    break;
                default:
                    throw new InvalidArgumentException('Unsupported schema type for ' . $key);
            }
        }
        return $output;
    }
}

class Validator
{
    public static function imageGeneration(): Schema
    {
        return new Schema([
            'prompt' => (new SchemaField('string'))->required()->min(1)->max(500),
            'width' => (new SchemaField('int'))->default(768)->min(256)->max(1024),
            'height' => (new SchemaField('int'))->default(768)->min(256)->max(1024),
            'steps' => (new SchemaField('int'))->default(30)->min(5)->max(150),
            'guidance' => (new SchemaField('float'))->default(7.5)->minFloat(1.0)->maxFloat(30.0),
            'seed' => (new SchemaField('mixed'))->default(null)->nullable(),
        ]);
    }
}